<?php
session_start();
require_once '../config/db.php';
require_once '../includes/auth_check.php';
require_once '../includes/functions.php';

// Check if user has permission to view supplier orders
$supplierPermissions = [
    'view_suppliers', 'manage_suppliers', 'add_supplier', 'edit_supplier', 'delete_supplier',
    'search_suppliers', 'process_supplier_payment', 'add_supplier_item'
];

if (!hasAnyPermission($supplierPermissions) && !$isSuperAdmin) {
    $_SESSION['error'] = "You don't have permission to view supplier orders.";
    header('Location: ../pages/dashboard.php');
    exit;
}

if (!isset($_GET['order_id']) || !is_numeric($_GET['order_id'])) {
    die('Invalid order ID');
}

$order_id = intval($_GET['order_id']);

try {
    // Get order details
    $stmt = $pdo->prepare("
        SELECT 
            so.*,
            s.name as supplier_name,
            s.contact_person,
            s.phone as supplier_phone,
            s.email as supplier_email,
            s.address as supplier_address,
            u.username as created_by_username,
            u.full_name as created_by_full_name
        FROM supplier_orders so
        JOIN suppliers s ON so.supplier_id = s.id
        JOIN users u ON so.created_by = u.id
        WHERE so.id = ?
    ");
    $stmt->execute([$order_id]);
    $order = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$order) {
        die('Order not found');
    }
    
    // Get order items
    $stmt = $pdo->prepare("
        SELECT 
            soi.*,
            p.name as product_name,
            p.description as product_description,
            c.name as category_name
        FROM supplier_order_items soi
        JOIN products p ON soi.product_id = p.id
        LEFT JOIN categories c ON p.category_id = c.id
        WHERE soi.order_id = ?
        ORDER BY p.name
    ");
    $stmt->execute([$order_id]);
    $items = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get store information
    $stmt = $pdo->prepare("
        SELECT setting_key, setting_value 
        FROM store_settings 
        WHERE setting_key IN ('store_name', 'store_address', 'store_phone', 'store_email')
    ");
    $stmt->execute();
    $storeSettings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
    
} catch (Exception $e) {
    die('Error loading order: ' . $e->getMessage());
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Supplier Order #<?php echo $order['id']; ?> - Print</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            font-size: 12px;
            line-height: 1.4;
            color: #333;
            margin: 0;
            padding: 20px;
        }
        
        .header {
            text-align: center;
            border-bottom: 2px solid #333;
            padding-bottom: 20px;
            margin-bottom: 30px;
        }
        
        .company-name {
            font-size: 24px;
            font-weight: bold;
            margin-bottom: 5px;
        }
        
        .company-details {
            font-size: 14px;
            color: #666;
        }
        
        .order-title {
            font-size: 20px;
            font-weight: bold;
            margin: 20px 0;
            text-align: center;
        }
        
        .order-info {
            display: flex;
            justify-content: space-between;
            margin-bottom: 30px;
        }
        
        .info-section {
            width: 48%;
        }
        
        .info-section h3 {
            font-size: 16px;
            margin-bottom: 10px;
            color: #333;
            border-bottom: 1px solid #ccc;
            padding-bottom: 5px;
        }
        
        .info-row {
            margin-bottom: 8px;
        }
        
        .info-label {
            font-weight: bold;
            display: inline-block;
            width: 120px;
        }
        
        .items-table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 20px;
        }
        
        .items-table th,
        .items-table td {
            border: 1px solid #ddd;
            padding: 8px;
            text-align: left;
        }
        
        .items-table th {
            background-color: #f5f5f5;
            font-weight: bold;
        }
        
        .items-table tr:nth-child(even) {
            background-color: #f9f9f9;
        }
        
        .status-badge {
            padding: 3px 8px;
            border-radius: 12px;
            font-size: 10px;
            font-weight: bold;
            text-transform: uppercase;
        }
        
        .status-pending { background-color: #fff3cd; color: #856404; }
        .status-approved { background-color: #d1ecf1; color: #0c5460; }
        .status-completed { background-color: #d4edda; color: #155724; }
        .status-cancelled { background-color: #f8d7da; color: #721c24; }
        .status-partial { background-color: #ffeaa7; color: #2d3436; }
        .status-received { background-color: #d4edda; color: #155724; }
        
        .summary {
            margin-top: 30px;
            padding: 15px;
            background-color: #f8f9fa;
            border: 1px solid #dee2e6;
        }
        
        .footer {
            margin-top: 50px;
            padding-top: 20px;
            border-top: 1px solid #ccc;
            text-align: center;
            font-size: 10px;
            color: #666;
        }
        
        @media print {
            body { margin: 0; padding: 15px; }
            .no-print { display: none; }
        }
    </style>
</head>
<body>
    <div class="no-print" style="margin-bottom: 20px;">
        <button onclick="window.print()" style="padding: 10px 20px; background: #007bff; color: white; border: none; border-radius: 5px; cursor: pointer;">
            Print Order
        </button>
        <button onclick="window.close()" style="padding: 10px 20px; background: #6c757d; color: white; border: none; border-radius: 5px; cursor: pointer; margin-left: 10px;">
            Close
        </button>
    </div>

    <div class="header">
        <div class="company-name"><?php echo htmlspecialchars($storeSettings['store_name'] ?? 'POS System'); ?></div>
        <div class="company-details">
            <?php if (!empty($storeSettings['store_address'])): ?>
                <?php echo htmlspecialchars($storeSettings['store_address']); ?><br>
            <?php endif; ?>
            <?php if (!empty($storeSettings['store_phone'])): ?>
                Phone: <?php echo htmlspecialchars($storeSettings['store_phone']); ?>
            <?php endif; ?>
            <?php if (!empty($storeSettings['store_email'])): ?>
                | Email: <?php echo htmlspecialchars($storeSettings['store_email']); ?>
            <?php endif; ?>
        </div>
    </div>

    <div class="order-title">
        SUPPLIER ORDER #<?php echo $order['id']; ?>
    </div>

    <div class="order-info">
        <div class="info-section">
            <h3>Order Information</h3>
            <div class="info-row">
                <span class="info-label">Order ID:</span>
                #<?php echo $order['id']; ?>
            </div>
            <div class="info-row">
                <span class="info-label">Order Date:</span>
                <?php echo date('F j, Y', strtotime($order['order_date'])); ?>
            </div>
            <div class="info-row">
                <span class="info-label">Expected Delivery:</span>
                <?php echo $order['expected_delivery_date'] ? date('F j, Y', strtotime($order['expected_delivery_date'])) : 'Not specified'; ?>
            </div>
            <div class="info-row">
                <span class="info-label">Status:</span>
                <span class="status-badge status-<?php echo $order['status']; ?>">
                    <?php echo ucfirst($order['status']); ?>
                </span>
            </div>
            <div class="info-row">
                <span class="info-label">Created By:</span>
                <?php echo htmlspecialchars($order['created_by_username']); ?>
            </div>
            <div class="info-row">
                <span class="info-label">Created At:</span>
                <?php echo date('F j, Y g:i A', strtotime($order['created_at'])); ?>
            </div>
        </div>

        <div class="info-section">
            <h3>Supplier Information</h3>
            <div class="info-row">
                <span class="info-label">Name:</span>
                <?php echo htmlspecialchars($order['supplier_name']); ?>
            </div>
            <?php if ($order['contact_person']): ?>
            <div class="info-row">
                <span class="info-label">Contact Person:</span>
                <?php echo htmlspecialchars($order['contact_person']); ?>
            </div>
            <?php endif; ?>
            <?php if ($order['supplier_phone']): ?>
            <div class="info-row">
                <span class="info-label">Phone:</span>
                <?php echo htmlspecialchars($order['supplier_phone']); ?>
            </div>
            <?php endif; ?>
            <?php if ($order['supplier_email']): ?>
            <div class="info-row">
                <span class="info-label">Email:</span>
                <?php echo htmlspecialchars($order['supplier_email']); ?>
            </div>
            <?php endif; ?>
            <?php if ($order['supplier_address']): ?>
            <div class="info-row">
                <span class="info-label">Address:</span>
                <?php echo nl2br(htmlspecialchars($order['supplier_address'])); ?>
            </div>
            <?php endif; ?>
        </div>
    </div>

    <?php if ($order['notes']): ?>
    <div style="margin-bottom: 20px;">
        <h3>Order Notes</h3>
        <p style="background: #f8f9fa; padding: 10px; border-left: 4px solid #007bff;">
            <?php echo nl2br(htmlspecialchars($order['notes'])); ?>
        </p>
    </div>
    <?php endif; ?>

    <table class="items-table">
        <thead>
            <tr>
                <th>Product</th>
                <th>Category</th>
                <th>Ordered Qty</th>
                <th>Received Qty</th>
                <th>Status</th>
                <th>Notes</th>
            </tr>
        </thead>
        <tbody>
            <?php 
            $total_ordered = 0;
            $total_received = 0;
            foreach ($items as $item): 
                $total_ordered += $item['quantity'];
                $total_received += $item['received_quantity'];
            ?>
            <tr>
                <td>
                    <strong><?php echo htmlspecialchars($item['product_name']); ?></strong>
                    <?php if ($item['product_description']): ?>
                        <br><small style="color: #666;"><?php echo htmlspecialchars($item['product_description']); ?></small>
                    <?php endif; ?>
                </td>
                <td><?php echo htmlspecialchars($item['category_name'] ?? '-'); ?></td>
                <td style="text-align: center;"><?php echo $item['quantity']; ?></td>
                <td style="text-align: center;"><?php echo $item['received_quantity']; ?></td>
                <td style="text-align: center;">
                    <span class="status-badge status-<?php echo $item['status']; ?>">
                        <?php echo ucfirst($item['status']); ?>
                    </span>
                </td>
                <td><?php echo htmlspecialchars($item['notes'] ?? '-'); ?></td>
            </tr>
            <?php endforeach; ?>
        </tbody>
    </table>

    <div class="summary">
        <div style="display: flex; justify-content: space-between;">
            <div>
                <strong>Order Summary:</strong><br>
                Total Items Ordered: <?php echo $total_ordered; ?><br>
                Total Items Received: <?php echo $total_received; ?><br>
                Completion Rate: <?php echo $total_ordered > 0 ? round(($total_received / $total_ordered) * 100, 1) : 0; ?>%
            </div>
            <div style="text-align: right;">
                <strong>Print Information:</strong><br>
                Printed on: <?php echo date('F j, Y g:i A'); ?><br>
                Printed by: <?php echo htmlspecialchars($_SESSION['username'] ?? 'System'); ?>
            </div>
        </div>
    </div>

    <div class="footer">
        This is a computer-generated document. No signature is required.
    </div>

    <script>
        // Auto-focus for printing
        window.addEventListener('load', function() {
            setTimeout(function() {
                window.focus();
            }, 100);
        });
    </script>
</body>
</html> 